////////////////////////////////////////////////////////////////////
//
// This fragment shader is used to punch holes in an image thereby
// allowing waht is behind it to be seen.  The shader is controled
// by a number of parameters that may be specified in a VGHD scene
// file using
//
//   uniform: spacing, vec4, nx, ny, fx, fy
//   uniform: holes,   vec4, shape, hole_opacity, material_opacity, spare
//   uniform: skew,    vec4, row_skew, col_skew, hole_skew_x, hole_skew_y
//
// For details of these parameters see the comments below.
//
///////////////////////////////////////////////////////////////////

// Primary inputs to the shader.

uniform float u_Elapsed;    // The elapsed time in seconds
uniform vec2  u_WindowSize; // Window dimensions in pixels

uniform sampler2D image;    // The image to be perforated

// Control parameters.  These get initialised to values specified in
// the .scn file.  They  are  grouped into three sets of parameters,
// each a vec4, rather than a larger number of individual parameters
// for convenience in the .scn file. The components of each vec4 are
// assigned specific roles by #define statements below.

uniform vec4 spacing; // Controls hole spacing and size
uniform vec4 holes;   // Controls hole shape and opacity
uniform vec4 skew;    // Controls hole alignment

///////////////////////////////////////////////////////////////////

// HOLE_COUNT is the number of perforations in the X and Y
// directions and HOLE_SIZE their dimensions as  fractions
// of the distances between them.

#define HOLE_COUNT vec2(spacing.x,spacing.y)
#define HOLE_SIZE  vec2(spacing.z,spacing.w)/HOLE_COUNT

// SHAPE controls the shape of the perforations.
//
//     0.0    - No perforation
//  0.0 - 1.0 - Diamond with incurving sides
//     1.0    - Diamond with straight sides
//  1.0 - 1.5 - Diamond with out curving sides
//  1.5 - 2.0 - Rugby ball or eye shaped
//     2.0    - Elipse
//    > 2.0   - Super elipse
// 
// by the time SHAPE reaches 10.0 the shape is very  close
// to rectangular and any values greater than 20.0 will be
// interpreted as requests for exact rectangles.  Negative
// values are also allowed and give similar shapes but the
// mapping from SHAPE to shape is less easy to describe as
// the numerical values for such things as Diamond are not
// such simple numbers, also the centers of the shapes are
// shifted.

#define SHAPE holes.x

// The opacities of the holes and of the material  between
// them are controlled by the following parameters. Normal
// values for HOLE_OPACITY will be 0.0, or close to it, so 
// the holes will be transparent and MATERIAL_OPACITY will
// normaly be close to 1.0 so that away from the holes the
// material will be opaque or nearly so - but other values
// can also be used to get different effects.

#define HOLE_OPACITY     holes.y
#define MATERIAL_OPACITY holes.z

// LINE_SKEW controls the relative alignment  of  adjacent
// rows of holes, a value of 0.0 means that the holes will
// be aligned and increasing its value will cause them  to
// be misaligned until it reaches 1.0 when they  are  once
// more fully aligned. 

#define LINE_SKEW vec2(skew.x,skew.y)

// HOLE_SKEW controls how much each hole is skewed. Values
// other than 0.0 cause the holes to be distorted so  that
// rectangles become parallelograms and the effect on  eye
// shapes etc. includes changing their orientation.

#define HOLE_SKEW vec2(skew.z,skew.w)

////////////////////////////////////////////////////////////////////////////////

void main ( void )
 {
   // Rescale the pixel's coordinates to be fractions of
   // the current window's dimensions.  Also  flip the Y
   // coordinate so that the image will be upright.
   
   vec2 uv = gl_FragCoord.xy / u_WindowSize;
   uv.y = 1.0 - uv.y;

   // Get skewed coordinates.  In the skewed coordinates
   // the axes are along the rows and columns  of  holes
   // but these need not be orthogonal.

   vec2 q = uv + LINE_SKEW*uv.yx;

   // Get the position of the center of the nearest hole
   // its size and its shape.  The effects of a negative
   // value for SHAPE can be seen  by  uncommenting  the
   // line that conditionaly negates shape.  If  this is
   // done use small values for NX and NY, e.g. 5.0, and
   // large values for FX and FY, e.g. 0.4,  so that the
   // the shapes of the holes can be clearly seen.

   vec2 a = 1.0/HOLE_COUNT;        // Hole spacing
   vec2 c = mod(q,a) - 0.5*a;      // Nearest center
   vec2 d = HOLE_SIZE*0.5;         // Hole size

   float shape = SHAPE;             // Hole shape
   // if(uv.x>0.5) shape *= -1.0;   // Test mode

   // Reverse the effect of the  LINE_SKEW otherwise  it
   // produces an unwanted effct on the hole's shape and
   // orientation and at the same time apply any  wanted
   // skew to the shape.

   c = c - (LINE_SKEW-HOLE_SKEW)*c.yx;

   // Determine if in perforation or not. Extreme values
   // of shape are treated as exact rectangles, which is
   // what they approximate anyway.

   bool in_a_hole;

   if ( shape > 20.0 )
    { in_a_hole = all ( lessThan(abs(c),d) );
    }
   else
    { 
      float e = pow(c.x,shape) + pow(c.y,shape);
      float f = pow(d.x,shape) + pow(d.y,shape);
      in_a_hole = e < f;
    }

   // Set the colour of the pixel depending on  whether 
   // it is a hole or not.  Typically HOLE_OPACITY will
   // be 0.0 and the hole will be fully transparent and
   // MATERIAL_OPACITY will be high for total or almost
   // total opacity outside of the holes.

   if ( in_a_hole )
    { gl_FragColor = vec4 ( vec3(0.0), HOLE_OPACITY );
    }
   else
    { gl_FragColor = vec4 ( texture2D(image,uv).xyz, MATERIAL_OPACITY );
    }

 }